<?php

    namespace App\Repositories\Eloquent;

    use App\Exceptions\GeneralException;
    use App\Models\Country;
    use App\Models\CustomSendingServer;
    use App\Models\SendingServer;
    use App\Models\SendingServerBasedPricingPlans;
    use App\Models\User;
    use App\Repositories\Contracts\SendingServerRepository;
    use Exception;
    use Illuminate\Support\Arr;
    use Illuminate\Support\Facades\DB;
    use Throwable;

    class EloquentSendingServerRepository extends EloquentBaseRepository implements SendingServerRepository
    {
        /**
         * EloquentSendingServerRepository constructor.
         */
        public function __construct(SendingServer $sendingServer)
        {
            parent::__construct($sendingServer);
        }

        /**
         * Store Sending server
         *
         *
         *
         * @throws GeneralException
         */
        public function store(array $input, array $options = []): SendingServer
        {
            $insertKeys = array_keys($this->allSendingServer()[$input['settings']] ?? []);

            /** @var SendingServer $sendingServer */
            $sendingServer = $this->make(Arr::only($input, $insertKeys));

            $sendingServer->status  = true;
            $sendingServer->user_id = auth()->id();

            if ( ! $this->save($sendingServer)) {
                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            }


            if (config('app.gateway_wise_billing') && ! blank($options)) {

                $countryList = $input['country'];

                $countryList = $countryList == 0
                    ? Country::where('status', 1)->pluck('id')->toArray()
                    : (array) $countryList;

                if ( ! empty($countryList)) {
                    foreach ($countryList as $country) {

                        $existing = SendingServerBasedPricingPlans::where('sending_server', $sendingServer->id)->where('country_id', $country)->first();

                        if ( ! empty($existing)) {
                            continue;
                        }

                        SendingServerBasedPricingPlans::create([
                            'country_id'     => $country,
                            'sending_server' => $sendingServer->id,
                            'status'         => 1,
                            'options'        => json_encode($options),
                        ]);
                    }

                }

            }

            return $sendingServer;
        }


        /**
         * All Sending Servers
         *
         * @return array[]
         */
        public function allSendingServer(): array
        {
            return [
                SendingServer::TYPE_TWILIO => [
                    'name'            => SendingServer::TYPE_TWILIO,
                    'settings'        => SendingServer::TYPE_TWILIO,
                    'account_sid'     => 'account_sid',
                    'auth_token'      => 'auth_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => true,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TWILIOCOPILOT => [
                    'name'            => SendingServer::TYPE_TWILIOCOPILOT,
                    'settings'        => SendingServer::TYPE_TWILIOCOPILOT,
                    'account_sid'     => 'account_sid',
                    'auth_token'      => 'auth_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_EASYSENDSMS => [
                    'name'            => SendingServer::TYPE_EASYSENDSMS,
                    'settings'        => SendingServer::TYPE_EASYSENDSMS,
                    'api_link'        => 'https://api.easysendsms.app/bulksms',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CHEAPGLOBALSMS => [
                    'name'            => SendingServer::TYPE_CHEAPGLOBALSMS,
                    'settings'        => SendingServer::TYPE_CHEAPGLOBALSMS,
                    'api_link'        => 'https://cheapglobalsms.com/api_v1',
                    'username'        => 'sub_account',
                    'password'        => 'sub_account_pass',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CLICKATELLTOUCH => [
                    'name'            => SendingServer::TYPE_CLICKATELLTOUCH,
                    'settings'        => SendingServer::TYPE_CLICKATELLTOUCH,
                    'api_link'        => 'https://platform.clickatell.com/messages/http/send',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CLICKATELLCENTRAL => [
                    'name'            => SendingServer::TYPE_CLICKATELLCENTRAL,
                    'settings'        => SendingServer::TYPE_CLICKATELLCENTRAL,
                    'api_link'        => 'https://api.clickatell.com/http/sendmsg',
                    'username'        => 'user_name',
                    'password'        => 'password',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ROUTEMOBILE => [
                    'name'            => SendingServer::TYPE_ROUTEMOBILE,
                    'settings'        => SendingServer::TYPE_ROUTEMOBILE,
                    'api_link'        => 'https://api.rmlconnect.net/bulksms/bulksms',
                    'username'        => 'user_name',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXTLOCAL => [
                    'name'            => SendingServer::TYPE_TEXTLOCAL,
                    'settings'        => SendingServer::TYPE_TEXTLOCAL,
                    'api_link'        => 'https://api.txtlocal.com/send/',
                    'mms_api_link'    => 'https://api.txtlocal.com/send_mms/',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_PLIVO => [
                    'name'            => SendingServer::TYPE_PLIVO,
                    'settings'        => SendingServer::TYPE_PLIVO,
                    'auth_id'         => 'auth_id',
                    'auth_token'      => 'auth_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_PLIVOPOWERPACK => [
                    'name'            => SendingServer::TYPE_PLIVOPOWERPACK,
                    'settings'        => SendingServer::TYPE_PLIVOPOWERPACK,
                    'auth_id'         => 'auth_id',
                    'auth_token'      => 'auth_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSGLOBAL => [
                    'name'            => SendingServer::TYPE_SMSGLOBAL,
                    'settings'        => SendingServer::TYPE_SMSGLOBAL,
                    'api_link'        => 'https://api.smsglobal.com/http-api.php',
                    'mms_api_link'    => 'https://api.smsglobal.com/mms/sendmms.php',
                    'username'        => 'user_name',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BULKSMS => [
                    'name'            => SendingServer::TYPE_BULKSMS,
                    'settings'        => SendingServer::TYPE_BULKSMS,
                    'api_link'        => 'https://api.bulksms.com/v1/messages',
                    'username'        => 'user_name',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_VONAGE => [
                    'name'            => SendingServer::TYPE_VONAGE,
                    'settings'        => SendingServer::TYPE_VONAGE,
                    'api_link'        => 'https://rest.nexmo.com/sms/json',
                    'api_key'         => 'api_key',
                    'api_secret'      => 'api_secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_INFOBIP => [
                    'name'            => SendingServer::TYPE_INFOBIP,
                    'settings'        => SendingServer::TYPE_INFOBIP,
                    'api_link'        => 'https://89g329.api.infobip.com/sms/2/text/advanced',
                    'voice_api_link'  => 'https://api.infobip.com/tts/3/single',
                    'api_key'         => 'api_key',
                    'c1'              => 'indiaDltPrincipalEntityId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_1S2U => [
                    'name'            => SendingServer::TYPE_1S2U,
                    'settings'        => SendingServer::TYPE_1S2U,
                    'api_link'        => 'https://api.1s2u.io/bulksms',
                    'username'        => 'user_name',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MESSAGEBIRD => [
                    'name'            => SendingServer::TYPE_MESSAGEBIRD,
                    'settings'        => SendingServer::TYPE_MESSAGEBIRD,
                    'api_link'        => 'https://api.bird.com/workspaces/{workspaceId}/channels/{channelId}/messages',
                    'access_key'      => 'access_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_AMAZONSNS => [
                    'name'            => SendingServer::TYPE_AMAZONSNS,
                    'settings'        => SendingServer::TYPE_AMAZONSNS,
                    'access_key'      => 'access_key',
                    'secret_access'   => 'secret_access',
                    'region'          => 'region',
                    'sms_type'        => 'Transactional',
                    'c1'              => '',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TYNTEC => [
                    'name'            => SendingServer::TYPE_TYNTEC,
                    'settings'        => SendingServer::TYPE_TYNTEC,
                    'api_link'        => 'https://api.tyntec.com/messaging/v1/sms',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WHATSAPPCHATAPI => [
                    'name'            => SendingServer::TYPE_WHATSAPPCHATAPI,
                    'settings'        => SendingServer::TYPE_WHATSAPPCHATAPI,
                    'api_link'        => 'https://eu8.chat-api.com/instance105654',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => true,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_KARIXIO => [
                    'name'            => SendingServer::TYPE_KARIXIO,
                    'settings'        => SendingServer::TYPE_KARIXIO,
                    'api_link'        => 'https://api.karix.io/message/',
                    'auth_id'         => 'auth_id',
                    'auth_token'      => 'auth_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SIGNALWIRE => [
                    'name'            => SendingServer::TYPE_SIGNALWIRE,
                    'settings'        => SendingServer::TYPE_SIGNALWIRE,
                    'api_link'        => 'https://example.signalwire.com',
                    'api_token'       => 'api_token',
                    'project_id'      => 'project_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TELNYX => [
                    'name'            => SendingServer::TYPE_TELNYX,
                    'settings'        => SendingServer::TYPE_TELNYX,
                    'api_link'        => 'https://api.telnyx.com/v2/messages',
                    'voice_api_link'  => 'https://api.telnyx.com/v2/texml/calls',
                    'api_key'         => 'api_key',
                    'c1'              => 'messaging_profile_id',
                    'c2'              => 'messaging_connection_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TELNYXNUMBERPOOL => [
                    'name'            => SendingServer::TYPE_TELNYXNUMBERPOOL,
                    'settings'        => SendingServer::TYPE_TELNYXNUMBERPOOL,
                    'api_link'        => 'https://api.telnyx.com/v2/messages/number_pool',
                    'api_key'         => 'api_key',
                    'c1'              => 'messaging_profile_id',
                    'c2'              => 'messaging_connection_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BANDWIDTH => [
                    'name'            => SendingServer::TYPE_BANDWIDTH,
                    'settings'        => SendingServer::TYPE_BANDWIDTH,
                    'api_link'        => 'https://messaging.bandwidth.com/api/v2/users/account_id/messages',
                    'api_token'       => 'api_token',
                    'api_secret'      => 'api_secret',
                    'application_id'  => 'application_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMPP => [
                    'name'            => SendingServer::TYPE_SMPP,
                    'settings'        => SendingServer::TYPE_SMPP,
                    'api_link'        => 'IP/DOMAIN',
                    'port'            => 'PORT',
                    'username'        => 'SYSTEM ID/Username',
                    'password'        => 'Password',
                    'source_addr_ton' => '5',
                    'source_addr_npi' => '0',
                    'dest_addr_ton'   => '1',
                    'dest_addr_npi'   => '0',
                    'c1'              => null,
                    'c2'              => 'WWW',
                    'schedule'        => true,
                    'type'            => 'smpp',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                /*
        |--------------------------------------------------------------------------
        | Added in version 3.0.1
        |--------------------------------------------------------------------------
        |
        |
        |
        */

                SendingServer::TYPE_ROUTEENET => [
                    'name'            => SendingServer::TYPE_ROUTEENET,
                    'settings'        => SendingServer::TYPE_ROUTEENET,
                    'api_link'        => 'https://connect.routee.net/sms',
                    'application_id'  => 'application_id',
                    'api_secret'      => 'application_secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_HUTCHLK => [
                    'name'            => SendingServer::TYPE_HUTCHLK,
                    'settings'        => SendingServer::TYPE_HUTCHLK,
                    'api_link'        => 'https://bsms.hutch.lk/',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TELETOPIASMS => [
                    'name'            => SendingServer::TYPE_TELETOPIASMS,
                    'settings'        => SendingServer::TYPE_TELETOPIASMS,
                    'api_link'        => 'https://api1.teletopiasms.no/gateway/v3/plain',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BROADCASTERMOBILE => [
                    'name'            => SendingServer::TYPE_BROADCASTERMOBILE,
                    'settings'        => SendingServer::TYPE_BROADCASTERMOBILE,
                    'api_link'        => 'https://api.broadcastermobile.com/brdcstr-endpoint-web/services/messaging',
                    'api_token'       => 'api_token',
                    'api_key'         => 'api_key',
                    'c1'              => 'country_code',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SOLUTIONS4MOBILES => [
                    'name'            => SendingServer::TYPE_SOLUTIONS4MOBILES,
                    'settings'        => SendingServer::TYPE_SOLUTIONS4MOBILES,
                    'auth_link'       => 'https://sms.solutions4mobiles.com/apis/auth',
                    'api_link'        => 'https://sms.solutions4mobiles.com/apis/sms/mt/v2/send',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BEEMAFRICA => [
                    'name'            => SendingServer::TYPE_BEEMAFRICA,
                    'settings'        => SendingServer::TYPE_BEEMAFRICA,
                    'api_link'        => 'https://apisms.beem.africa/v1/send',
                    'api_key'         => 'api_key',
                    'api_secret'      => 'api_secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BULKSMSONLINE => [
                    'name'            => SendingServer::TYPE_BULKSMSONLINE,
                    'settings'        => SendingServer::TYPE_BULKSMSONLINE,
                    'api_link'        => 'https://api.bulksmsonline.com:9090/smsapi',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_FLOWROUTE => [
                    'name'            => SendingServer::TYPE_FLOWROUTE,
                    'settings'        => SendingServer::TYPE_FLOWROUTE,
                    'api_link'        => 'https://api.flowroute.com/v2.1/messages',
                    'access_key'      => 'access_key',
                    'api_secret'      => 'secret_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WAAPI => [
                    'name'            => SendingServer::TYPE_WAAPI,
                    'settings'        => SendingServer::TYPE_WAAPI,
                    'api_link'        => 'https://waapi.net/api/sendMessageText',
                    'c1'              => 'instance_key',
                    'c2'              => 'jid',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ELITBUZZBD => [
                    'name'            => SendingServer::TYPE_ELITBUZZBD,
                    'settings'        => SendingServer::TYPE_ELITBUZZBD,
                    'api_link'        => 'https://msg.elitbuzz-bd.com/smsapi',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_GREENWEBBD => [
                    'name'            => SendingServer::TYPE_GREENWEBBD,
                    'settings'        => SendingServer::TYPE_GREENWEBBD,
                    'api_link'        => 'https://api.greenweb.com.bd/api.php?json',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_HABLAMEV2 => [
                    'name'            => SendingServer::TYPE_HABLAMEV2,
                    'settings'        => SendingServer::TYPE_HABLAMEV2,
                    'api_link'        => 'https://api101.hablame.co/api/sms/v2.1/send/',
                    'api_token'       => 'api_token',
                    'api_key'         => 'api_key',
                    'c1'              => 'account',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ZAMTELCOZM => [
                    'name'            => SendingServer::TYPE_ZAMTELCOZM,
                    'settings'        => SendingServer::TYPE_ZAMTELCOZM,
                    'api_link'        => 'https://bulksms.zamtel.co.zm/api/v2.1/action/send/',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CELLCAST => [
                    'name'            => SendingServer::TYPE_CELLCAST,
                    'settings'        => SendingServer::TYPE_CELLCAST,
                    'api_link'        => 'https://cellcast.com.au/api/v3/send-sms',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_AFRICASTALKING => [
                    'name'            => SendingServer::TYPE_AFRICASTALKING,
                    'settings'        => SendingServer::TYPE_AFRICASTALKING,
                    'api_link'        => 'https://api.africastalking.com/version1/messaging',
                    'api_key'         => 'api_key',
                    'username'        => 'username',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CAIHCOM => [
                    'name'            => SendingServer::TYPE_CAIHCOM,
                    'settings'        => SendingServer::TYPE_CAIHCOM,
                    'api_link'        => 'https://sms.caihcom.com/sms/send',
                    'api_token'       => 'api_token',
                    'c1'              => 'channel_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_KECCELSMS => [
                    'name'            => SendingServer::TYPE_KECCELSMS,
                    'settings'        => SendingServer::TYPE_KECCELSMS,
                    'api_link'        => 'http://161.97.92.251:22099/message',
                    'application_id'  => 'id',
                    'password'        => 'pass',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_JOHNSONCONNECT => [
                    'name'            => SendingServer::TYPE_JOHNSONCONNECT,
                    'settings'        => SendingServer::TYPE_JOHNSONCONNECT,
                    'api_link'        => 'http://161.117.182.177:8080/api/sms/mtsend',
                    'api_key'         => 'app_key',
                    'api_secret'      => 'secret_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SPEEDAMOBILE => [
                    'name'            => SendingServer::TYPE_SPEEDAMOBILE,
                    'settings'        => SendingServer::TYPE_SPEEDAMOBILE,
                    'api_link'        => 'https://api.smsala.com/api/SendSMS',
                    'auth_id'         => 'api_id',
                    'password'        => 'api_password',
                    'sms_type'        => 'Promotional',
                    'c1'              => 'indiaDltPrincipalEntityId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSALA => [
                    'name'            => SendingServer::TYPE_SMSALA,
                    'settings'        => SendingServer::TYPE_SMSALA,
                    'api_link'        => 'https://api.smsala.com/api/SendSMS',
                    'auth_id'         => 'api_id',
                    'password'        => 'api_password',
                    'sms_type'        => 'Promotional',
                    'c1'              => 'indiaDltPrincipalEntityId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXT2WORLD => [
                    'name'            => SendingServer::TYPE_TEXT2WORLD,
                    'settings'        => SendingServer::TYPE_TEXT2WORLD,
                    'api_link'        => 'https://text2world.com/sendsms/sendsms.php',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ENABLEX => [
                    'name'            => SendingServer::TYPE_ENABLEX,
                    'settings'        => SendingServer::TYPE_ENABLEX,
                    'api_link'        => 'https://api.enablex.io/sms/v1/messages/',
                    'application_id'  => 'APP ID',
                    'api_key'         => 'APP Key',
                    'c1'              => 'Campaign ID',
                    'c2'              => 'Template ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SPOOFSEND => [
                    'name'            => SendingServer::TYPE_SPOOFSEND,
                    'settings'        => SendingServer::TYPE_SPOOFSEND,
                    'api_link'        => 'https://user.spoofsend.com/smsAPI',
                    'api_key'         => 'api_key',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ALHAJSMS => [
                    'name'            => SendingServer::TYPE_ALHAJSMS,
                    'settings'        => SendingServer::TYPE_ALHAJSMS,
                    'api_link'        => 'https://login.haj-sms.com/smsAPI',
                    'api_key'         => 'api_key',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SENDROIDULTIMATE => [
                    'name'            => SendingServer::TYPE_SENDROIDULTIMATE,
                    'settings'        => SendingServer::TYPE_SENDROIDULTIMATE,
                    'api_link'        => 'https://yourapplicationurl.com/smsAPI',
                    'api_key'         => 'api_key',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_REALSMS => [
                    'name'            => SendingServer::TYPE_REALSMS,
                    'settings'        => SendingServer::TYPE_REALSMS,
                    'api_link'        => 'https://api1.realsms.cc/API/',
                    'username'        => 'Username',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CALLR => [
                    'name'            => SendingServer::TYPE_CALLR,
                    'settings'        => SendingServer::TYPE_CALLR,
                    'api_link'        => 'https://api.callr.com/rest/v1.1/sms',
                    'username'        => 'api_login',
                    'password'        => 'api_password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SKYETEL => [
                    'name'            => SendingServer::TYPE_SKYETEL,
                    'settings'        => SendingServer::TYPE_SKYETEL,
                    'api_link'        => 'https://sms.skyetel.com/v1/out',
                    'account_sid'     => 'account_sid',
                    'api_secret'      => 'api_secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_LTR => [
                    'name'            => SendingServer::TYPE_LTR,
                    'settings'        => SendingServer::TYPE_LTR,
                    'api_link'        => 'https://sms.lrt.com.pk/api/sms-single-or-bulk-api.php',
                    'username'        => 'username',
                    'password'        => 'password',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BULKSMSPLANS => [
                    'name'            => SendingServer::TYPE_BULKSMSPLANS,
                    'settings'        => SendingServer::TYPE_BULKSMSPLANS,
                    'api_link'        => 'https://bulksmsplans.com/api/send_sms',
                    'auth_id'         => 'API ID',
                    'password'        => 'API Password',
                    'route'           => 'Transactional',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SINCH => [
                    'name'            => SendingServer::TYPE_SINCH,
                    'settings'        => SendingServer::TYPE_SINCH,
                    'api_link'        => 'https://us.sms.api.sinch.com/xms/v1/{$service_plan_id}/batches',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_D7NETWORKS => [
                    'name'            => SendingServer::TYPE_D7NETWORKS,
                    'settings'        => SendingServer::TYPE_D7NETWORKS,
                    'api_link'        => 'https://api.d7networks.com/messages/v1/send',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CMCOM => [
                    'name'            => SendingServer::TYPE_CMCOM,
                    'settings'        => SendingServer::TYPE_CMCOM,
                    'api_link'        => 'https://gw.cmtelecom.com/v1.0/message',
                    'api_token'       => 'product_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'PitchWink' => [
                    'name'            => 'PitchWink',
                    'settings'        => 'PitchWink',
                    'api_link'        => 'https://pw-api.com/sms/v_4_00/postsms.aspx',
                    'c1'              => 'Credential',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'Wavy' => [
                    'name'            => 'Wavy',
                    'settings'        => 'Wavy',
                    'api_link'        => 'https://api-messaging.wavy.global/v1/send-sms',
                    'auth_token'      => 'authentication token',
                    'username'        => 'username',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'Solucoesdigitais' => [
                    'name'            => 'Solucoesdigitais',
                    'settings'        => 'Solucoesdigitais',
                    'api_link'        => 'https://sms.solucoesdigitais.cc/integracao/v2/envio_transacional',
                    'username'        => 'usuario',
                    'password'        => 'senha',
                    'c1'              => 'centro_custo_interno',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SmartVision'                  => [
                    'name'            => 'SmartVision',
                    'settings'        => 'SmartVision',
                    'api_link'        => 'https://customers.smsmarketing.ae/app/smsapi/index.php',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'ZipComIo'                     => [
                    'name'            => 'ZipComIo',
                    'settings'        => 'ZipComIo',
                    'api_link'        => 'https://api.zipcom.io/2019-01-23/messages',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'GlobalSMSCN'                  => [
                    'name'            => 'GlobalSMSCN',
                    'settings'        => 'GlobalSMSCN',
                    'api_link'        => 'https://sms.skylinelabs.cc:20004/v3/sendSms',
                    'api_key'         => 'API KEY',
                    'api_secret'      => 'API Secret',
                    'application_id'  => 'Application ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'Web2SMS237'                   => [
                    'name'            => 'Web2SMS237',
                    'settings'        => 'Web2SMS237',
                    'api_link'        => 'https://api.web2sms237.com/sms/send',
                    'api_key'         => 'User ID',
                    'api_secret'      => 'User Secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'BongaTech'                    => [
                    'name'            => 'BongaTech',
                    'settings'        => 'BongaTech',
                    'api_link'        => 'https://bulk.bongatech.co.ke/api/v1/send-basic-sms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'FloatSMS'                     => [
                    'name'            => 'FloatSMS',
                    'settings'        => 'FloatSMS',
                    'api_link'        => 'https://app.floatsms.com/api/send',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                SendingServer::TYPE_YOOAPI     => [
                    'name'            => 'YooAPI',
                    'settings'        => 'YooAPI',
                    'api_link'        => 'https://my.yooapi.com/api/send.php',
                    'c1'              => 'client_id',
                    'c2'              => 'instance_id',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'MaisSMS'                      => [
                    'name'            => 'MaisSMS',
                    'settings'        => 'MaisSMS',
                    'api_link'        => 'https://http.maissms.com.br/mt',
                    'api_token'       => 'Token',
                    'c1'              => 'parceiro_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                SendingServer::TYPE_EASYSMSXYZ => [
                    'name'            => SendingServer::TYPE_EASYSMSXYZ,
                    'settings'        => SendingServer::TYPE_EASYSMSXYZ,
                    'api_link'        => 'https://mojsms.io/mmsi/services/send.php',
                    'api_key'         => 'Api Key',
                    'device_id'       => 1,
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'Sozuri'                       => [
                    'name'            => 'Sozuri',
                    'settings'        => 'Sozuri',
                    'api_link'        => 'https://sozuri.net/api/v1/messaging',
                    'api_key'         => 'Api Key',
                    'project_id'      => 'Project',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'ExpertTexting'                => [
                    'name'            => 'ExpertTexting',
                    'settings'        => 'ExpertTexting',
                    'api_link'        => 'https://www.experttexting.com/ExptRestApi/sms/json/Message/Send',
                    'username'        => 'Username',
                    'api_key'         => 'api_key',
                    'api_secret'      => 'api_secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'Ejoin'                        => [
                    'name'            => 'Ejoin',
                    'settings'        => 'Ejoin',
                    'api_link'        => 'http://103.114.97.146/goip_post_sms.html',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'BulkSMSNigeria'               => [
                    'name'            => 'BulkSMSNigeria',
                    'settings'        => 'BulkSMSNigeria',
                    'api_link'        => 'https://www.bulksmsnigeria.com/api/v2/sms',
                    'api_token'       => 'API Token',
                    'c1'              => 'direct-refund',
                    'c2'              => 'hosted',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SendSMSGate' => [
                    'name'            => 'SendSMSGate',
                    'settings'        => 'SendSMSGate',
                    'api_link'        => 'https://cloud.sendsmsgate.com/sendsms.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'Gateway360' => [
                    'name'            => 'Gateway360',
                    'settings'        => 'Gateway360',
                    'api_link'        => 'https://api.gateway360.com/api/3.0/sms/send',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'AjuraTech'  => [
                    'name'            => 'AjuraTech',
                    'settings'        => 'AjuraTech',
                    'api_link'        => 'https://smpp.ajuratech.com:7790/sendtext',
                    'api_key'         => 'API KEY',
                    'api_secret'      => 'Secret KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SMSCloudCI' => [
                    'name'            => 'SMSCloudCI',
                    'settings'        => 'SMSCloudCI',
                    'api_link'        => 'https://api.smscloud.ci/v1/campaigns',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'LifetimeSMS'              => [
                    'name'            => 'LifetimeSMS',
                    'settings'        => 'LifetimeSMS',
                    'api_link'        => 'https://lifetimesms.com/plain',
                    'api_token'       => 'API TOKEN',
                    'api_secret'      => 'API SECRET',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'PARATUS'                  => [
                    'name'            => 'PARATUS',
                    'settings'        => 'PARATUS',
                    'api_link'        => 'https://[playSMS_domain_or_url]/index.php',
                    'username'        => 'Username',
                    'api_token'       => 'Webservices Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'MOOVCI'                   => [
                    'name'            => 'MOOVCI',
                    'settings'        => 'MOOVCI',
                    'api_link'        => 'http://41.191.68.213:80/apiSms/http/sendsms/',
                    'api_key'         => 'api key',
                    'c1'              => 'Login',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'LeTexto'                  => [
                    'name'            => 'LeTexto',
                    'settings'        => 'LeTexto',
                    'api_link'        => 'https://api.letexto.com/v1/campaigns',
                    'api_token'       => 'api token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'SMSCarrierEU'             => [
                    'name'            => 'SMSCarrierEU',
                    'settings'        => 'SMSCarrierEU',
                    'api_link'        => 'https://smsc.i-digital-m.com/smsgw/sendsms.php',
                    'username'        => 'username',
                    'password'        => 'password',
                    'type'            => 'http',
                    'schedule'        => true,
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'MSMPusher'                => [
                    'name'            => 'MSMPusher',
                    'settings'        => 'MSMPusher',
                    'api_link'        => 'https://api.msmpusher.net/v1/send',
                    'c1'              => 'Private Key',
                    'c2'              => 'Public Key',
                    'type'            => 'http',
                    'schedule'        => true,
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                SendingServer::TYPE_TXTRIA => [
                    'name'            => SendingServer::TYPE_TXTRIA,
                    'settings'        => SendingServer::TYPE_TXTRIA,
                    'api_link'        => 'https://txtria.com/api/sendsms',
                    'mms_api_link'    => 'https://txtria.com/api/sendsms',
                    'c1'              => 'System ID',
                    'auth_token'      => 'Auth Token',
                    'type'            => 'http',
                    'schedule'        => true,
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WHATSENDER => [
                    'name'            => SendingServer::TYPE_WHATSENDER,
                    'settings'        => SendingServer::TYPE_WHATSENDER,
                    'api_link'        => 'https://api.whatsender.io/v1/messages',
                    'api_token'       => 'api_token',
                    'device_id'       => 'device_id',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => true,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'Gatewayapi' => [
                    'name'            => 'Gatewayapi',
                    'settings'        => 'Gatewayapi',
                    'api_link'        => 'https://gatewayapi.com/rest/mtsms',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'CamooCM' => [
                    'name'            => 'CamooCM',
                    'settings'        => 'CamooCM',
                    'api_link'        => 'https://api.camoo.cm/v1/sms.json',
                    'api_key'         => 'API KEY',
                    'api_secret'      => 'API SECRET',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SemySMS' => [
                    'name'            => 'SemySMS',
                    'settings'        => 'SemySMS',
                    'api_link'        => 'https://semysms.net/api/3/user.php',
                    'api_token'       => 'Api Token',
                    'device_id'       => 'Device',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'Xmsway' => [
                    'name'            => 'Xmsway',
                    'settings'        => 'Xmsway',
                    'api_link'        => 'https://wht.xmsway.com/api/send.php',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'BurstSMS' => [
                    'name'            => 'BurstSMS',
                    'settings'        => 'BurstSMS',
                    'api_link'        => 'https://api.transmitsms.com/send-sms.json',
                    'api_key'         => 'api_key',
                    'api_secret'      => 'api_secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'Inteliquent' => [
                    'name'            => 'Inteliquent',
                    'settings'        => 'Inteliquent',
                    'api_link'        => 'https://messagebroker.inteliquent.com/msgbroker/rest/publishMessages',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'VisionUp'    => [
                    'name'            => 'VisionUp',
                    'settings'        => 'VisionUp',
                    'api_link'        => 'http://142.93.78.16/api/sms',
                    'username'        => 'username',
                    'password'        => 'password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'FHMCloud' => [
                    'name'            => 'FHMCloud',
                    'settings'        => 'FHMCloud',
                    'api_link'        => 'https://web2sms.fhm.cloud/public/api/v3/sms/send',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SMSTO' => [
                    'name'            => 'SMSTO',
                    'settings'        => 'SMSTO',
                    'api_link'        => 'https://api.sms.to/sms/send',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'TextBelt'  => [
                    'name'            => 'TextBelt',
                    'settings'        => 'TextBelt',
                    'api_link'        => 'https://textbelt.com/text',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'IntelTele' => [
                    'name'            => 'IntelTele',
                    'settings'        => 'IntelTele',
                    'api_link'        => 'https://api.sms.intel-tele.com/message/send/',
                    'username'        => 'Username',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'GatewaySa' => [
                    'name'            => 'GatewaySa',
                    'settings'        => 'GatewaySa',
                    'api_link'        => 'https://api.gateway.sa/api/v2/SendSMS',
                    'api_key'         => 'api_key',
                    'c1'              => 'Client ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'OnBuka' => [
                    'name'            => 'OnBuka',
                    'settings'        => 'OnBuka',
                    'api_link'        => 'https://api.onbuka.com/v3/sendSms',
                    'api_key'         => 'api_key',
                    'api_secret'      => 'api_secret',
                    'application_id'  => 'application_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'BulkGate' => [
                    'name'            => 'BulkGate',
                    'settings'        => 'BulkGate',
                    'api_link'        => 'https://portal.bulkgate.com/api/1.0/simple/transactional',
                    'api_token'       => 'api_token',
                    'application_id'  => 'application_id',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SMSVas' => [
                    'name'            => 'SMSVas',
                    'settings'        => 'SMSVas',
                    'api_link'        => 'https://smsvas.vlserv.com/VLSMSPlatformResellerAPI/NewSendingAPI/api/SMSSender/SendSMSWithDLR',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'IconGlobalCoUK' => [
                    'name'            => 'IconGlobalCoUK',
                    'settings'        => 'IconGlobalCoUK',
                    'api_link'        => 'https://sms.iconglobal.co.uk/API/SendSMS',
                    'username'        => 'Username',
                    'application_id'  => 'API ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'SendPulse'      => [
                    'name'            => 'SendPulse',
                    'settings'        => 'SendPulse',
                    'api_link'        => 'https://api.sendpulse.com/sms/send',
                    'c1'              => 'Client ID',
                    'api_secret'      => 'Client Secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SpewHub'  => [
                    'name'            => 'SpewHub',
                    'settings'        => 'SpewHub',
                    'api_link'        => 'https://spewhub.com/api/sms/broadcast',
                    'c1'              => 'Licence Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'CCSSMS'   => [
                    'name'            => 'CCSSMS',
                    'settings'        => 'CCSSMS',
                    'api_link'        => 'http://62.67.222.27:8001/api',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                'TeleSign' => [
                    'name'            => 'TeleSign',
                    'settings'        => 'TeleSign',
                    'api_link'        => 'https://rest-ww.telesign.com/v1/messaging',
                    'c1'              => 'Customer ID',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'ClearComMX' => [
                    'name'            => 'ClearComMX',
                    'settings'        => 'ClearComMX',
                    'api_link'        => 'https://sms.clearcom.mx/api/v2/sms/send',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'CyberGateLK' => [
                    'name'            => 'CyberGateLK',
                    'settings'        => 'CyberGateLK',
                    'api_link'        => 'https://smsgw-node.cybergate.lk:10002/v1/external/sms/s/send',
                    'api_token'       => 'Auth TOKEN',
                    'auth_key'        => 'User Auth Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'LuxSMS' => [
                    'name'            => 'LuxSMS',
                    'settings'        => 'LuxSMS',
                    'api_link'        => 'https://api.luxsms.net/api/SendSMS',
                    'c1'              => 'API ID',
                    'password'        => 'API Password',
                    'sms_type'        => 'Transactional',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'MidasAppBr' => [
                    'name'            => 'MidasAppBr',
                    'settings'        => 'MidasAppBr',
                    'api_link'        => 'https://midas.app.br/send-template',
                    'api_key'         => 'API Key',
                    'c1'              => 'Footer Message',
                    'c2'              => 'Template 1',
                    'c3'              => 'Template 2',
                    'c4'              => 'Button 1',
                    'c5'              => 'Button 2',
                    'c6'              => 'Button 3',
                    'c7'              => 'URL',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                'SMSCrab'                     => [
                    'name'            => 'SMSCrab',
                    'settings'        => 'SMSCrab',
                    'api_link'        => 'https://smscrab.com/abc/api/v3/sms/send',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.4*/
                SendingServer::TYPE_SAFARICOM => [
                    'name'            => SendingServer::TYPE_SAFARICOM,
                    'settings'        => SendingServer::TYPE_SAFARICOM,
                    'api_link'        => 'https://dtsvc.safaricom.com:8480/api/',
                    'username'        => 'Auth UserName',
                    'password'        => 'Auth Password',
                    'c1'              => 'Campaign UserName',
                    'project_id'      => 'Package ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_FACILITAMOVEL => [
                    'name'            => SendingServer::TYPE_FACILITAMOVEL,
                    'settings'        => SendingServer::TYPE_FACILITAMOVEL,
                    'api_link'        => 'https://api.facilitamovel.com.br/api/simpleSend.ft',
                    'username'        => 'UserName',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSDELIVERER => [
                    'name'            => SendingServer::TYPE_SMSDELIVERER,
                    'settings'        => SendingServer::TYPE_SMSDELIVERER,
                    'api_link'        => 'http://192.168.43.210:8888/',
                    'username'        => 'UserName',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ROUNDSMS => [
                    'name'            => SendingServer::TYPE_ROUNDSMS,
                    'settings'        => SendingServer::TYPE_ROUNDSMS,
                    'api_link'        => 'https://roundsms.com/api/sendhttp.php',
                    'auth_key'        => 'Auth Key',
                    'route'           => 'Route',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_YOSMS => [
                    'name'            => SendingServer::TYPE_YOSMS,
                    'settings'        => SendingServer::TYPE_YOSMS,
                    'api_link'        => 'https://smgw1.yo.co.ug:9100/sendsms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DIGINTRA => [
                    'name'            => SendingServer::TYPE_DIGINTRA,
                    'settings'        => SendingServer::TYPE_DIGINTRA,
                    'api_link'        => 'http://164.52.196.82:6005/api/v2/SendSMS',
                    'api_key'         => 'API KEY',
                    'c1'              => 'Client ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ALLMYSMS => [
                    'name'            => SendingServer::TYPE_ALLMYSMS,
                    'settings'        => SendingServer::TYPE_ALLMYSMS,
                    'api_link'        => 'https://api.allmysms.com/sms/send',
                    'auth_key'        => 'Auth Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ESOLUTIONS => [
                    'name'            => SendingServer::TYPE_ESOLUTIONS,
                    'settings'        => SendingServer::TYPE_ESOLUTIONS,
                    'api_link'        => 'https://mobile.esolutions.co.zw/bmg/api/single ',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_GUPSHUPIO => [
                    'name'            => SendingServer::TYPE_GUPSHUPIO,
                    'settings'        => SendingServer::TYPE_GUPSHUPIO,
                    'api_link'        => 'https://api.gupshup.io/sm/api/v1/msg',
                    'api_key'         => 'api_key',
                    'c1'              => 'Gupshup App Name',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => true,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SEMAPHORE => [
                    'name'            => SendingServer::TYPE_SEMAPHORE,
                    'settings'        => SendingServer::TYPE_SEMAPHORE,
                    'api_link'        => 'https://semaphore.co/api/v4/messages',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ESTORESMS => [
                    'name'            => SendingServer::TYPE_ESTORESMS,
                    'settings'        => SendingServer::TYPE_ESTORESMS,
                    'api_link'        => 'https://www.estoresms.com/smsapi.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_GOIP => [
                    'name'            => SendingServer::TYPE_GOIP,
                    'settings'        => SendingServer::TYPE_GOIP,
                    'api_link'        => 'http://31.223.4.35/default/en_US/send.html',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MAILJET => [
                    'name'            => SendingServer::TYPE_MAILJET,
                    'settings'        => SendingServer::TYPE_MAILJET,
                    'api_link'        => 'https://api.mailjet.com/v4/sms-send',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ADVANCEMSGSYS => [
                    'name'            => SendingServer::TYPE_ADVANCEMSGSYS,
                    'settings'        => SendingServer::TYPE_ADVANCEMSGSYS,
                    'api_link'        => 'http://78.108.164.69:8080/websmpp/websms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_UIPAPP => [
                    'name'            => SendingServer::TYPE_UIPAPP,
                    'settings'        => SendingServer::TYPE_UIPAPP,
                    'api_link'        => 'https://www.uipsms.com/app/api/international-sms/submit',
                    'user_token'      => 'User Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSFRL => [
                    'name'            => SendingServer::TYPE_SMSFRL,
                    'settings'        => SendingServer::TYPE_SMSFRL,
                    'api_link'        => 'https://sms.frl/api/orders',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_IMARTGROUP => [
                    'name'            => SendingServer::TYPE_IMARTGROUP,
                    'settings'        => SendingServer::TYPE_IMARTGROUP,
                    'api_link'        => 'https://smsportal.imartgroup.co.tz/app/smsapi/index.php',
                    'api_key'         => 'API Key',
                    'route'           => '8',
                    'c1'              => '1',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.5*/

                SendingServer::TYPE_GOSMSFUN => [
                    'name'            => SendingServer::TYPE_GOSMSFUN,
                    'settings'        => SendingServer::TYPE_GOSMSFUN,
                    'api_link'        => 'https://api.gosms.fun/sms/v3/send',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXT_CALIBUR => [
                    'name'            => SendingServer::TYPE_TEXT_CALIBUR,
                    'settings'        => SendingServer::TYPE_TEXT_CALIBUR,
                    'api_link'        => 'https://api.text-calibur.com/v1/sms/send',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WHATSAPP => [
                    'name'            => 'WhatsApp By Facebook',
                    'settings'        => SendingServer::TYPE_WHATSAPP,
                    'api_link'        => 'https://your-webapp-hostname:your-webapp-port/v1/messages',
                    'access_token'    => 'Access Token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SAVEWEBHOSTNET => [
                    'name'            => SendingServer::TYPE_SAVEWEBHOSTNET,
                    'settings'        => SendingServer::TYPE_SAVEWEBHOSTNET,
                    'api_link'        => 'https://www.sms.savewebhost.net/api/v3/sms/send',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_FAST2SMS => [
                    'name'            => SendingServer::TYPE_FAST2SMS,
                    'settings'        => SendingServer::TYPE_FAST2SMS,
                    'api_link'        => 'https://www.fast2sms.com/dev/bulkV2',
                    'api_key'         => 'API Key',
                    'c1'              => 'DLT Entity ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MSG91 => [
                    'name'            => SendingServer::TYPE_MSG91,
                    'settings'        => SendingServer::TYPE_MSG91,
                    'api_link'        => 'https://api.msg91.com/api/v5/flow/',
                    'otp_api_link'    => 'https://control.msg91.com/api/v5/otp',
                    'auth_key'        => 'Auth Key',
                    'c1'              => 'Flow ID',
                    'c2'              => 'Template ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TELEAPI => [
                    'name'            => SendingServer::TYPE_TELEAPI,
                    'settings'        => SendingServer::TYPE_TELEAPI,
                    'api_link'        => 'https://api.teleapi.net/sms/send',
                    'mms_api_link'    => 'https://api.teleapi.net/mms/send',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BUDGETSMS => [
                    'name'            => SendingServer::TYPE_BUDGETSMS,
                    'settings'        => SendingServer::TYPE_BUDGETSMS,
                    'api_link'        => 'https://api.budgetsms.net/sendsms/',
                    'username'        => 'Username',
                    'c1'              => 'User ID',
                    'c2'              => 'Handle',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_OZONEDESK => [
                    'name'            => SendingServer::TYPE_OZONEDESK,
                    'settings'        => SendingServer::TYPE_OZONEDESK,
                    'api_link'        => 'https://send.ozonedesk.com/api/v2/send.php',
                    'api_key'         => 'API Key',
                    'c1'              => 'User ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SKEBBY => [
                    'name'            => SendingServer::TYPE_SKEBBY,
                    'settings'        => SendingServer::TYPE_SKEBBY,
                    'api_link'        => 'https://api.skebby.it/API/v1.0/REST/sms',
                    'api_key'         => 'User Key',
                    'access_token'    => 'Access Token',
                    'c1'              => 'Message Type',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BULKREPLY => [
                    'name'            => SendingServer::TYPE_BULKREPLY,
                    'settings'        => SendingServer::TYPE_BULKREPLY,
                    'api_link'        => 'https://bulkreply.com/api/send',
                    'access_token'    => 'Access Token',
                    'c1'              => 'Instance ID',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BULkSMS4BTC => [
                    'name'            => SendingServer::TYPE_BULkSMS4BTC,
                    'settings'        => SendingServer::TYPE_BULkSMS4BTC,
                    'api_link'        => 'https://Bulksms4btc.ru/api/user/instance',
                    'api_key'         => 'API',
                    'c1'              => 'Variable',
                    'c2'              => 'Variable',
                    'c3'              => 'Variable',
                    'c4'              => 'Variable',
                    'c5'              => 'Variable',
                    'c6'              => 'Variable',
                    'c7'              => 'Variable',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.6*/

                SendingServer::TYPE_NIMBUZ => [
                    'name'            => SendingServer::TYPE_NIMBUZ,
                    'settings'        => SendingServer::TYPE_NIMBUZ,
                    'api_link'        => 'http://162.55.22.113/api/pushsmsjson.php',
                    'username'        => 'User',
                    'api_key'         => 'API KEY',
                    'c1'              => 'EntityID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MOBITECH => [
                    'name'            => SendingServer::TYPE_MOBITECH,
                    'settings'        => SendingServer::TYPE_MOBITECH,
                    'api_link'        => 'https://api.mobitechtechnologies.com/sms/sendsms',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TRUBLOQ => [
                    'name'            => SendingServer::TYPE_TRUBLOQ,
                    'settings'        => SendingServer::TYPE_TRUBLOQ,
                    'api_link'        => 'https://api.mobitechtechnologies.com/sms/sendsms',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_HOSTPINNACLE => [
                    'name'            => SendingServer::TYPE_HOSTPINNACLE,
                    'settings'        => SendingServer::TYPE_HOSTPINNACLE,
                    'api_link'        => 'https://smsportal.hostpinnacle.co.ke/SMSApi/send',
                    'username'        => 'User Name',
                    'password'        => 'Password',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_LANKABELL => [
                    'name'            => SendingServer::TYPE_LANKABELL,
                    'settings'        => SendingServer::TYPE_LANKABELL,
                    'api_link'        => 'https://smsm.lankabell.com:4090/Sms.svc/SecureSendSms',
                    'api_key'         => 'API Header KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_PICKYASSIST => [
                    'name'            => SendingServer::TYPE_PICKYASSIST,
                    'settings'        => SendingServer::TYPE_PICKYASSIST,
                    'api_link'        => 'https://pickyassist.com/app/api/v2/push',
                    'api_token'       => 'API Token',
                    'application_id'  => 'Application ID',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ZORRA => [
                    'name'            => SendingServer::TYPE_ZORRA,
                    'settings'        => SendingServer::TYPE_ZORRA,
                    'api_link'        => 'https://my.zorra.com/api/',
                    'username'        => 'User Email',
                    'password'        => 'User Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_HOTMOBILE => [
                    'name'            => SendingServer::TYPE_HOTMOBILE,
                    'settings'        => SendingServer::TYPE_HOTMOBILE,
                    'api_link'        => 'https://painel.hotmobile.com.br/SendAPI/Send.aspx',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_YUPCHAT => [
                    'name'            => SendingServer::TYPE_YUPCHAT,
                    'settings'        => SendingServer::TYPE_YUPCHAT,
                    'api_link'        => 'https://api.yup.chat/v1/sms/messages',
                    'c1'              => 'SEU ID',
                    'api_token'       => 'SEU Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_8x8 => [
                    'name'            => SendingServer::TYPE_8x8,
                    'settings'        => SendingServer::TYPE_8x8,
                    'api_link'        => 'https://sms.8x8.com/api/v1/subaccounts/{subAccountId}/messages',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_FONOIP => [
                    'name'            => SendingServer::TYPE_FONOIP,
                    'settings'        => SendingServer::TYPE_FONOIP,
                    'api_link'        => 'https://navarrete.fonoip.com/api/send_sms',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WAZONE => [
                    'name'            => SendingServer::TYPE_WAZONE,
                    'settings'        => SendingServer::TYPE_WAZONE,
                    'api_link'        => 'https://api.wa-omnipro.com/send',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_QOOLIZE => [
                    'name'            => SendingServer::TYPE_QOOLIZE,
                    'settings'        => SendingServer::TYPE_QOOLIZE,
                    'api_link'        => 'http://162.55.18.97:4848/sendsms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_EBULKSMS => [
                    'name'              => SendingServer::TYPE_EBULKSMS,
                    'settings'          => SendingServer::TYPE_EBULKSMS,
                    'api_link'          => 'https://api.ebulksms.com/sendsms.json',
                    'whatsapp_api_link' => 'https://api.ebulksms.com/sendwhatsapp.json',
                    'username'          => 'Username',
                    'api_key'           => 'API Key',
                    'schedule'          => true,
                    'type'              => 'http',
                    'two_way'           => false,
                    'plain'             => true,
                    'mms'               => false,
                    'voice'             => false,
                    'whatsapp'          => true,
                    'viber'             => false,
                    'otp'               => false,
                    'sms_per_request'   => 1,
                    'quota_value'       => 500,
                    'quota_base'        => 1,
                    'quota_unit'        => 'minute',
                ],

                SendingServer::TYPE_CLICKSEND => [
                    'name'            => SendingServer::TYPE_CLICKSEND,
                    'settings'        => SendingServer::TYPE_CLICKSEND,
                    'username'        => 'Username',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WHATSAPPBYTEMPLATE => [
                    'name'            => 'Facebook WhatsApp By Template',
                    'settings'        => SendingServer::TYPE_WHATSAPPBYTEMPLATE,
                    'api_link'        => 'https://graph.facebook.com/v15.0/{whatsapp-business-account-ID}/messages',
                    'access_token'    => 'Access Token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ALIBABACLOUDSMS => [
                    'name'            => SendingServer::TYPE_ALIBABACLOUDSMS,
                    'settings'        => SendingServer::TYPE_ALIBABACLOUDSMS,
                    'api_link'        => 'dysmsapi.ap-southeast-1.aliyuncs.com',
                    'access_key'      => 'Access Key ID',
                    'secret_access'   => 'Access Secret',
                    'region'          => 'RegionID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSMODE => [
                    'name'            => SendingServer::TYPE_SMSMODE,
                    'settings'        => SendingServer::TYPE_SMSMODE,
                    'api_link'        => 'https://rest.smsmode.com/sms/v1/messages',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TECHCORE => [
                    'name'            => SendingServer::TYPE_TECHCORE,
                    'settings'        => SendingServer::TYPE_TECHCORE,
                    'api_link'        => 'https://sms.techcore.biz/API/SendSMS',
                    'username'        => 'Username',
                    'application_id'  => 'API ID',
                    'access_token'    => 'AccessToken',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ORANGE => [
                    'name'            => SendingServer::TYPE_ORANGE,
                    'settings'        => SendingServer::TYPE_ORANGE,
                    'api_link'        => 'https://api.orange.com/',
                    'c1'              => 'Basic OE51WWgxOHRjc0Y0Q0J6eVNyQkd2dXo6MBqZkJyc0o3bg==',
                    'c2'              => 'DevPhoneNumber',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MMSCONSOLE => [
                    'name'            => SendingServer::TYPE_MMSCONSOLE,
                    'settings'        => SendingServer::TYPE_MMSCONSOLE,
                    'api_link'        => 'https://mms-console.com/API/V2/send',
                    'c1'              => 'Email',
                    'api_token'       => 'Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BMSGLOBAL => [
                    'name'            => SendingServer::TYPE_BMSGLOBAL,
                    'settings'        => SendingServer::TYPE_BMSGLOBAL,
                    'api_link'        => 'https://app.wa2sales.com/api/api.php',
                    'api_token'       => 'Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_GBESTSMS => [
                    'name'            => SendingServer::TYPE_GBESTSMS,
                    'settings'        => SendingServer::TYPE_GBESTSMS,
                    'api_link'        => 'https://gbestsms.com/component/spc/',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SILVERSTREET => [
                    'name'            => SendingServer::TYPE_SILVERSTREET,
                    'settings'        => SendingServer::TYPE_SILVERSTREET,
                    'api_link'        => 'https://ic1.silverstreet.com/send.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_GLINTSMS => [
                    'name'            => SendingServer::TYPE_GLINTSMS,
                    'settings'        => SendingServer::TYPE_GLINTSMS,
                    'api_link'        => 'https://glintsms.com/login/sms/api',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DATAGIFTING => [
                    'name'            => SendingServer::TYPE_DATAGIFTING,
                    'settings'        => SendingServer::TYPE_DATAGIFTING,
                    'api_link'        => 'https://datagifting.com.ng/api/sms.php',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSHTTPREVE => [
                    'name'            => SendingServer::TYPE_SMSHTTPREVE,
                    'settings'        => SendingServer::TYPE_SMSHTTPREVE,
                    'api_link'        => 'http://149.20.190.5:3524/sendtext',
                    'api_key'         => 'api key',
                    'api_secret'      => 'Secret key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BULKSMSPROVIDERNG => [
                    'name'            => SendingServer::TYPE_BULKSMSPROVIDERNG,
                    'settings'        => SendingServer::TYPE_BULKSMSPROVIDERNG,
                    'api_link'        => 'https://sms.bulksmsprovider.ng/api/',
                    'voice_api_link'  => 'https://sms.bulksmsprovider.ng/api/',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_OZONESMS => [
                    'name'            => SendingServer::TYPE_OZONESMS,
                    'settings'        => SendingServer::TYPE_OZONESMS,
                    'api_link'        => 'https://smsozone.com/api/mt/SendSMS',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'sms_type'        => 'Transactional',
                    'route'           => 'RouteID',
                    'c1'              => 'indiaDltPrincipalEntityId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_NIGERIABULKSMS => [
                    'name'            => SendingServer::TYPE_NIGERIABULKSMS,
                    'settings'        => SendingServer::TYPE_NIGERIABULKSMS,
                    'api_link'        => 'https://portal.nigeriabulksms.com/api/',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_AIRTELINDIA => [
                    'name'            => SendingServer::TYPE_AIRTELINDIA,
                    'settings'        => SendingServer::TYPE_AIRTELINDIA,
                    'api_link'        => 'https://openapi.airtel.in/gateway/airtel-xchange/v2/execute/workflow',
                    'voice_api_link'  => 'https://openapi.airtel.in/gateway/airtel-xchange/v2/execute/workflow',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'c1'              => 'CallFlowId',
                    'c2'              => 'CustomerId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.7*/

                SendingServer::TYPE_SMSAPI => [
                    'name'            => SendingServer::TYPE_SMSAPI,
                    'settings'        => SendingServer::TYPE_SMSAPI,
                    'api_link'        => 'https://api.smsapi.com/sms.do',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSAPIONLINE => [
                    'name'            => SendingServer::TYPE_SMSAPIONLINE,
                    'settings'        => SendingServer::TYPE_SMSAPIONLINE,
                    'api_link'        => 'https://api.sms-api.online/send',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TERMII => [
                    'name'            => SendingServer::TYPE_TERMII,
                    'settings'        => SendingServer::TYPE_TERMII,
                    'api_link'        => 'https://api.ng.termii.com/api/sms/send',
                    'api_key'         => 'API Key',
                    'c1'              => 'Channel',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_VOXIMPLANT => [
                    'name'            => SendingServer::TYPE_VOXIMPLANT,
                    'settings'        => SendingServer::TYPE_VOXIMPLANT,
                    'api_link'        => 'https://api.voximplant.com/platform_api/SendSmsMessage',
                    'api_key'         => 'API Key',
                    'c1'              => 'Account ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CLIQSMS => [
                    'name'            => SendingServer::TYPE_CLIQSMS,
                    'settings'        => SendingServer::TYPE_CLIQSMS,
                    'api_link'        => 'https://plus.cliqsms.com/sendsms.php',
                    'username'        => 'User',
                    'password'        => 'Password',
                    'c1'              => 'DND',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSVEND => [
                    'name'            => SendingServer::TYPE_SMSVEND,
                    'settings'        => SendingServer::TYPE_SMSVEND,
                    'api_link'        => 'https://api.smsvend.com/v2/app/sms',
                    'c1'              => 'Email',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_PMCSMS => [
                    'name'            => SendingServer::TYPE_PMCSMS,
                    'settings'        => SendingServer::TYPE_PMCSMS,
                    'api_link'        => 'https://pmcsms.com/api/v1/http.php',
                    'api_key'         => 'API Key',
                    'route'           => 'Route',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WA2SALES => [
                    'name'            => SendingServer::TYPE_WA2SALES,
                    'settings'        => SendingServer::TYPE_WA2SALES,
                    'api_link'        => 'https://app.wa2sales.com/api/api2.php',
                    'api_token'       => 'API Token',
                    'c1'              => 'Footer Message',
                    'c2'              => 'CallButton Text',
                    'c3'              => 'CallButton Value',
                    'c4'              => 'URLButton Text',
                    'c5'              => 'URLButton Value',
                    'c6'              => 'QuickReplyButton Text',
                    'c7'              => 'QuickReplyButton Text',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_INTERAKT => [
                    'name'            => SendingServer::TYPE_INTERAKT,
                    'settings'        => SendingServer::TYPE_INTERAKT,
                    'api_link'        => 'https://api.interakt.ai/v1/public/message/',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSAFRICANG => [
                    'name'            => SendingServer::TYPE_SMSAFRICANG,
                    'settings'        => SendingServer::TYPE_SMSAFRICANG,
                    'api_link'        => 'https://www.smsafricang.com/api/resellers/addons/sendsms2.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MOBILESMSNG => [
                    'name'            => SendingServer::TYPE_MOBILESMSNG,
                    'settings'        => SendingServer::TYPE_MOBILESMSNG,
                    'api_link'        => 'https://www.mobilesmsng.com/api/resellers/addons/sendsms2.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BSGWORLD => [
                    'name'            => SendingServer::TYPE_BSGWORLD,
                    'settings'        => SendingServer::TYPE_BSGWORLD,
                    'api_link'        => 'https://api.bsg.world/rest/sms/create',
                    'api_key'         => 'X-API-KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.8*/
                SendingServer::TYPE_SNAPISMS => [
                    'name'            => SendingServer::TYPE_SNAPISMS,
                    'settings'        => SendingServer::TYPE_SNAPISMS,
                    'api_link'        => 'https://api.snapisms.io/api/v1/send',
                    'api_key'         => 'API KEY',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSEXPERIENCE => [
                    'name'            => SendingServer::TYPE_SMSEXPERIENCE,
                    'settings'        => SendingServer::TYPE_SMSEXPERIENCE,
                    'api_link'        => 'https://smsexperience.com/api/sms/sendsms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CHALLENGESMS => [
                    'name'            => SendingServer::TYPE_CHALLENGESMS,
                    'settings'        => SendingServer::TYPE_CHALLENGESMS,
                    'api_link'        => 'https://campaign.challengesms.com/api/send/sms',
                    'api_secret'      => 'Secret',
                    'device_id'       => 'Device ID',
                    'c1'              => 'Mode',
                    'c2'              => 'Gateway',
                    'c3'              => 'sim',
                    'c4'              => 'priority',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BLACKSMS => [
                    'name'            => SendingServer::TYPE_BLACKSMS,
                    'settings'        => SendingServer::TYPE_BLACKSMS,
                    'api_link'        => 'https://sm.blacksms.com.ng/api/v3/sms/send',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ULTIMATESMS => [
                    'name'            => SendingServer::TYPE_ULTIMATESMS,
                    'settings'        => SendingServer::TYPE_ULTIMATESMS,
                    'api_link'        => 'https://ultimatesms.codeglen.com/demo/api/http/sms/send',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MOCEANAPI => [
                    'name'            => SendingServer::TYPE_MOCEANAPI,
                    'settings'        => SendingServer::TYPE_MOCEANAPI,
                    'api_link'        => 'https://rest.moceanapi.com/rest/2/sms',
                    'api_key'         => 'API Key',
                    'api_secret'      => 'API Secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MESSAGGIO => [
                    'name'            => SendingServer::TYPE_MESSAGGIO,
                    'settings'        => SendingServer::TYPE_MESSAGGIO,
                    'api_link'        => 'https://msg.messaggio.com/api/v1/send',
                    'api_key'         => 'API Key',
                    'sms_type'        => 'Promotional',
                    'schedule'        => true,
                    'type'            => 'viber',
                    'two_way'         => true,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => true,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSURWAY => [
                    'name'            => SendingServer::TYPE_SMSURWAY,
                    'settings'        => SendingServer::TYPE_SMSURWAY,
                    'api_link'        => 'https://smsurway.com.ng/portal/api/login',
                    'otp_api_link'    => 'https://smsurway.com.ng/portal/api/send',
                    'c1'              => 'Email',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMARTSMSSOLUTIONS => [
                    'name'            => SendingServer::TYPE_SMARTSMSSOLUTIONS,
                    'settings'        => SendingServer::TYPE_SMARTSMSSOLUTIONS,
                    'api_link'        => 'https://app.smartsmssolutions.com/io/api/client/v1/sms/',
                    'voice_api_link'  => 'https://app.smartsmssolutions.com/io/api/client/v1/voiceotp/send/',
                    'otp_api_link'    => 'https://app.smartsmssolutions.com/io/api/client/v1/smsotp/send/',
                    'api_token'       => 'TOKEN',
                    'route'           => 'Routing',
                    'application_id'  => 'AppNameCode',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_VOICEANDTEXT => [
                    'name'            => SendingServer::TYPE_VOICEANDTEXT,
                    'settings'        => SendingServer::TYPE_VOICEANDTEXT,
                    'api_link'        => 'https://dnd.voiceandtext.com/customer/api/',
                    'voice_api_link'  => 'https://robocall.voiceandtext.com/aggoutcall.php',
                    'otp_api_link'    => 'https://voiceotp.voiceandtext.com/pushotp.php',
                    'username'        => 'Username',
                    'password'        => 'Password/APIAdmin',
                    'api_token'       => 'AdminToken',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ETROSS => [
                    'name'            => SendingServer::TYPE_ETROSS,
                    'settings'        => SendingServer::TYPE_ETROSS,
                    'api_link'        => 'http://68.14.237.90:8000/wg_api_v2/sendTask',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DINSTAR => [
                    'name'            => SendingServer::TYPE_DINSTAR,
                    'settings'        => SendingServer::TYPE_DINSTAR,
                    'api_link'        => 'https://gateway_ip/api/send_sms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'port'            => 'Port',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SIMPLETEXTING => [
                    'name'            => SendingServer::TYPE_SIMPLETEXTING,
                    'settings'        => SendingServer::TYPE_SIMPLETEXTING,
                    'api_link'        => 'https://api-app2.simpletexting.com/v2/api/messages',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WAVIX => [
                    'name'            => SendingServer::TYPE_WAVIX,
                    'settings'        => SendingServer::TYPE_WAVIX,
                    'api_link'        => 'https://api.wavix.com/v2/messages',
                    'application_id'  => 'App ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ARKESEL => [
                    'name'            => SendingServer::TYPE_ARKESEL,
                    'settings'        => SendingServer::TYPE_ARKESEL,
                    'api_link'        => 'https://sms.arkesel.com/sms/api',
                    'voice_api_link'  => 'https://sms.arkesel.com/api/v2/sms/voice/send',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DIAFAAN => [
                    'name'            => SendingServer::TYPE_DIAFAAN,
                    'settings'        => SendingServer::TYPE_DIAFAAN,
                    'api_link'        => 'http://144.91.88.62:9710',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'c1'              => '1',
                    'c2'              => '5',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_LAFRICAMOBILE => [
                    'name'            => SendingServer::TYPE_LAFRICAMOBILE,
                    'settings'        => SendingServer::TYPE_LAFRICAMOBILE,
                    'api_link'        => 'https://lamsms.lafricamobile.com/api',
                    'c1'              => 'Account ID',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.9*/

                SendingServer::TYPE_GUPSHUPIOTEMPLATE => [
                    'name'            => SendingServer::TYPE_GUPSHUPIOTEMPLATE,
                    'settings'        => SendingServer::TYPE_GUPSHUPIOTEMPLATE,
                    'api_link'        => 'https://api.gupshup.io/wa/api/v1/template/msg',
                    'api_key'         => 'api_key',
                    'c1'              => 'Gupshup App Name',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => true,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_AUDIENCEONE => [
                    'name'            => SendingServer::TYPE_AUDIENCEONE,
                    'settings'        => SendingServer::TYPE_AUDIENCEONE,
                    'api_link'        => 'https://calls.audienceone.net/mbilling/api.php',
                    'api_key'         => 'api_key',
                    'c1'              => 'PhoneBook',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSAERO => [
                    'name'            => SendingServer::TYPE_SMSAERO,
                    'settings'        => SendingServer::TYPE_SMSAERO,
                    'api_link'        => 'gate.smsaero.ru/v2/sms/send',
                    'api_key'         => 'api_key',
                    'c1'              => 'Email',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXTGRID => [
                    'name'            => SendingServer::TYPE_TEXTGRID,
                    'settings'        => SendingServer::TYPE_TEXTGRID,
                    'api_link'        => 'https://api.textgrid.com/2010-04-01/Accounts/{accountSid}/Messages.json',
                    'account_sid'     => 'Account Sid',
                    'auth_token'      => 'Auth Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TOPYING => [
                    'name'            => SendingServer::TYPE_TOPYING,
                    'settings'        => SendingServer::TYPE_TOPYING,
                    'api_link'        => 'http://8.219.185.254:20003/sendsms',
                    'c1'              => 'Account',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MASCOM => [
                    'name'            => SendingServer::TYPE_MASCOM,
                    'settings'        => SendingServer::TYPE_MASCOM,
                    'api_link'        => 'https://apiv2client.mascom.linksms.co.bw/v1/api/developers/send-sms',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MP => [
                    'name'            => SendingServer::TYPE_MP,
                    'settings'        => SendingServer::TYPE_MP,
                    'api_link'        => 'http://176.9.109.249:4848/sendsms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_360DIALOG => [
                    'name'            => SendingServer::TYPE_360DIALOG,
                    'settings'        => SendingServer::TYPE_360DIALOG,
                    'api_link'        => 'https://waba-v2.360dialog.io/messages',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BROADBASED => [
                    'name'            => SendingServer::TYPE_BROADBASED,
                    'settings'        => SendingServer::TYPE_BROADBASED,
                    'api_link'        => 'https://vas.broadbased.net/smsModule/sms/send/singleMessage',
                    'api_secret'      => 'API Secret',
                    'c1'              => 'User ID',
                    'sms_type'        => 'Promotional',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MOVIDER => [
                    'name'            => SendingServer::TYPE_MOVIDER,
                    'settings'        => SendingServer::TYPE_MOVIDER,
                    'api_link'        => 'https://api.movider.co/v1/sms',
                    'api_key'         => 'API Key',
                    'api_secret'      => 'API Secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ZENDER => [
                    'name'            => SendingServer::TYPE_ZENDER,
                    'settings'        => SendingServer::TYPE_ZENDER,
                    'api_link'        => 'https://sms.sociabo.com/api/send/sms',
                    'api_secret'      => 'API Secret',
                    'sms_type'        => 'devices',
                    'device_id'       => 'Device id',
                    'c1'              => 'Gateway id',
                    'c2'              => '1',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_EKOSMS => [
                    'name'            => SendingServer::TYPE_EKOSMS,
                    'settings'        => SendingServer::TYPE_EKOSMS,
                    'api_link'        => 'https://api-public.ekotech.cm/messages',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ALDEAMO => [
                    'name'            => SendingServer::TYPE_ALDEAMO,
                    'settings'        => SendingServer::TYPE_ALDEAMO,
                    'api_link'        => 'https://apitellit.aldeamo.com/SmsiWS/smsSendPost/',
                    'voice_api_link'  => 'https://apitellitvoice.aldeamo.com/voice/sendPost/',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'c1'              => 'Country Code',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TONKRA => [
                    'name'            => SendingServer::TYPE_TONKRA,
                    'settings'        => SendingServer::TYPE_TONKRA,
                    'api_link'        => 'https://api.tonkra.com/httpsms/send',
                    'c1'              => 'Client',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSDENVER => [
                    'name'            => SendingServer::TYPE_SMSDENVER,
                    'settings'        => SendingServer::TYPE_SMSDENVER,
                    'api_link'        => 'https://atlascommunications-api.com/sms/send',
                    'auth_token'      => 'Auth Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WAUSMS => [
                    'name'            => SendingServer::TYPE_WAUSMS,
                    'settings'        => SendingServer::TYPE_WAUSMS,
                    'api_link'        => 'https://dashboard.wausms.com/Api/rest/message',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DIDWW => [
                    'name'            => SendingServer::TYPE_DIDWW,
                    'settings'        => SendingServer::TYPE_DIDWW,
                    'api_link'        => 'https://sms-out.didww.com/outbound_messages',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_4JAWALY => [
                    'name'            => SendingServer::TYPE_4JAWALY,
                    'settings'        => SendingServer::TYPE_4JAWALY,
                    'api_link'        => 'https://api-sms.4jawaly.com/api/v1/account/area/sms/send',
                    'api_key'         => 'API Key',
                    'api_secret'      => 'API Secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSGATEWAYHUB => [
                    'name'            => SendingServer::TYPE_SMSGATEWAYHUB,
                    'settings'        => SendingServer::TYPE_SMSGATEWAYHUB,
                    'api_link'        => 'https://smsgatewayhub.com/api/mt/SendSms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'route'           => 'Route',
                    'sms_type'        => '1',
                    'c1'              => 'indiaDltPrincipalEntityId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SLING => [
                    'name'            => SendingServer::TYPE_SLING,
                    'settings'        => SendingServer::TYPE_SLING,
                    'api_link'        => 'https://app.sling.com.ng/api/v1/send-sms',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SLEENGSHORT => [
                    'name'            => SendingServer::TYPE_SLEENGSHORT,
                    'settings'        => SendingServer::TYPE_SLEENGSHORT,
                    'api_link'        => 'https://app.sleengshort.com/api/sms/send',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_EMISRI => [
                    'name'            => SendingServer::TYPE_EMISRI,
                    'settings'        => SendingServer::TYPE_EMISRI,
                    'api_link'        => 'https://app.emisri.com/API/SendSMS',
                    'api_key'         => 'API Key',
                    'api_token'       => 'API Token',
                    'username'        => 'Username',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MTN => [
                    'name'            => SendingServer::TYPE_MTN,
                    'settings'        => SendingServer::TYPE_MTN,
                    'api_link'        => 'https://api.mtn.com/v3/sms/messages/sms/outbound',
                    'c1'              => 'Client ID',
                    'api_secret'      => 'Client Secret',
                    'c2'              => 'Client Correlator Id',
                    'c3'              => 'Service Code',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MOBILETEXTALERTS => [
                    'name'            => SendingServer::TYPE_MOBILETEXTALERTS,
                    'settings'        => SendingServer::TYPE_MOBILETEXTALERTS,
                    'api_link'        => 'https://api.mobile-text-alerts.com/v3/send',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MSEGAT => [
                    'name'            => SendingServer::TYPE_MSEGAT,
                    'settings'        => SendingServer::TYPE_MSEGAT,
                    'api_link'        => 'https://www.msegat.com/gw/sendsms.php',
                    'username'        => 'Username',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_LABSMOBILE => [
                    'name'            => SendingServer::TYPE_LABSMOBILE,
                    'settings'        => SendingServer::TYPE_LABSMOBILE,
                    'api_link'        => 'https://api.labsmobile.com/json/send',
                    'username'        => 'Username',
                    'api_token'       => 'API TOKEN',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.10.0*/
                SendingServer::TYPE_WHGATE     => [
                    'name'            => SendingServer::TYPE_WHGATE,
                    'settings'        => SendingServer::TYPE_WHGATE,
                    'api_link'        => 'https://bot.whgate.com/api/send',
                    'c1'              => 'Instance ID',
                    'access_token'    => 'Access Token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],


                SendingServer::TYPE_WAWHGATE => [
                    'name'            => SendingServer::TYPE_WAWHGATE,
                    'settings'        => SendingServer::TYPE_WAWHGATE,
                    'api_link'        => 'https://wa.whgate.com/api/create-message',
                    'api_key'         => 'App Key',
                    'auth_key'        => 'Auth Key',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSLIVE247 => [
                    'name'            => SendingServer::TYPE_SMSLIVE247,
                    'settings'        => SendingServer::TYPE_SMSLIVE247,
                    'api_link'        => 'https://api.smslive247.com/api/v4/sms',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_LINKMOBILITY => [
                    'name'            => SendingServer::TYPE_LINKMOBILITY,
                    'settings'        => SendingServer::TYPE_LINKMOBILITY,
                    'api_link'        => 'https://n-eu.linkmobility.io',
                    'c1'              => 'Client ID',
                    'api_secret'      => 'Client Secret',
                    'c2'              => 'platformId',
                    'c3'              => 'platformPartnerId',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSCOUNTRY => [
                    'name'            => SendingServer::TYPE_SMSCOUNTRY,
                    'settings'        => SendingServer::TYPE_SMSCOUNTRY,
                    'api_link'        => 'https://restapi.smscountry.com/v0.1/Accounts/{authKey}/SMSes/',
                    'auth_key'        => 'Auth Key',
                    'auth_token'      => 'Auth Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXTBEE => [
                    'name'            => SendingServer::TYPE_TEXTBEE,
                    'settings'        => SendingServer::TYPE_TEXTBEE,
                    'api_link'        => 'https://api.textbee.dev/api/v1',
                    'api_key'         => 'API Key',
                    'device_id'       => 'Device ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_800COM => [
                    'name'            => SendingServer::TYPE_800COM,
                    'settings'        => SendingServer::TYPE_800COM,
                    'api_link'        => 'https://api.800.com/v1/messages',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_PHONECOM => [
                    'name'            => SendingServer::TYPE_PHONECOM,
                    'settings'        => SendingServer::TYPE_PHONECOM,
                    'api_link'        => 'https://api.phone.com/v4/accounts/{voip_id}/messages',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ULTRAMSG => [
                    'name'            => SendingServer::TYPE_ULTRAMSG,
                    'settings'        => SendingServer::TYPE_ULTRAMSG,
                    'access_token'    => 'Access Token',
                    'c1'              => 'instance id',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],


                SendingServer::TYPE_AFFILIATESMS => [
                    'name'            => SendingServer::TYPE_AFFILIATESMS,
                    'settings'        => SendingServer::TYPE_AFFILIATESMS,
                    'api_link'        => 'https://api.affiliate-sms.com/rest/sms/create',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSEEDGE => [
                    'name'            => SendingServer::TYPE_SMSEEDGE,
                    'settings'        => SendingServer::TYPE_SMSEEDGE,
                    'api_link'        => 'https://api.smsedge.com/v1/sms/send-single/',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DOTGO => [
                    'name'            => SendingServer::TYPE_DOTGO,
                    'settings'        => SendingServer::TYPE_DOTGO,
                    'api_link'        => 'https://api.dotgo.com/rcs/bot/v1/{botId}/messages',
                    'c1'              => 'Client ID',
                    'api_secret'      => 'Client Secret',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TELUCOMAPIS => [
                    'name'            => SendingServer::TYPE_TELUCOMAPIS,
                    'settings'        => SendingServer::TYPE_TELUCOMAPIS,
                    'api_link'        => 'https://backup.telucomapis.com/api2/index.php',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_NOTIFYRE => [
                    'name'            => SendingServer::TYPE_NOTIFYRE,
                    'settings'        => SendingServer::TYPE_NOTIFYRE,
                    'api_link'        => 'https://api.notifyre.com/sms/send',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_NOBELSMS => [
                    'name'            => SendingServer::TYPE_NOBELSMS,
                    'settings'        => SendingServer::TYPE_NOBELSMS,
                    'api_link'        => 'https://api.nobelsms.com/rest/send_sms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_FORDEMO => [
                    'name'            => SendingServer::TYPE_FORDEMO,
                    'settings'        => SendingServer::TYPE_FORDEMO,
                    'api_link'        => 'https://codeglen.com/demo',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => true,
                    'plain'           => true,
                    'mms'             => true,
                    'voice'           => true,
                    'whatsapp'        => true,
                    'viber'           => true,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SKYELINE => [
                    'name'            => SendingServer::TYPE_SKYELINE,
                    'settings'        => SendingServer::TYPE_SKYELINE,
                    'api_link'        => 'http://54.179.122.227:55004/goip_send_sms.html',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'port'            => '1',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                /*Version 3.11*/

                SendingServer::TYPE_OURSMS => [
                    'name'            => SendingServer::TYPE_OURSMS,
                    'settings'        => SendingServer::TYPE_OURSMS,
                    'api_link'        => 'https://api.oursms.com/api-a/msgs',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BROADNET => [
                    'name'            => SendingServer::TYPE_BROADNET,
                    'settings'        => SendingServer::TYPE_BROADNET,
                    'api_link'        => 'https://gw1sp.broadnet.me/index.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                SendingServer::TYPE_SEVENIO  => [
                    'name'            => SendingServer::TYPE_SEVENIO,
                    'settings'        => SendingServer::TYPE_SEVENIO,
                    'api_link'        => 'https://gateway.seven.io/api/sms',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                SendingServer::TYPE_MNOTIFY  => [
                    'name'            => SendingServer::TYPE_MNOTIFY,
                    'settings'        => SendingServer::TYPE_MNOTIFY,
                    'api_link'        => 'https://api.mnotify.com/api/sms/quick',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],
                SendingServer::TYPE_HUBTEL   => [
                    'name'            => SendingServer::TYPE_HUBTEL,
                    'settings'        => SendingServer::TYPE_HUBTEL,
                    'api_link'        => 'https://devp-sms03726-api.hubtel.com/v1/messages/send',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BEENET => [
                    'name'            => SendingServer::TYPE_BEENET,
                    'settings'        => SendingServer::TYPE_BEENET,
                    'api_link'        => 'https://13.234.255.157/sendsms/sendsms.php',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSWORLDPRO => [
                    'name'            => SendingServer::TYPE_SMSWORLDPRO,
                    'settings'        => SendingServer::TYPE_SMSWORLDPRO,
                    'api_link'        => 'https://smsworld.pro/api/envioApi',
                    'api_key'         => 'API Key',
                    'api_token'       => 'API Token',
                    'c1'              => 'API ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_UNITEL => [
                    'name'            => SendingServer::TYPE_UNITEL,
                    'settings'        => SendingServer::TYPE_UNITEL,
                    'auth_link'       => 'https://api.unisend.unitel.ao/imicampaignapi/resources/v3/oauth/token',
                    'api_link'        => 'https://api.unisend.unitel.ao/imicampaignapi/resources/v3/events',
                    'api_key'         => 'API Key',
                    'api_secret'      => 'API Secret',
                    'c1'              => 'Event ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TURKEYSMS => [
                    'name'            => SendingServer::TYPE_TURKEYSMS,
                    'settings'        => SendingServer::TYPE_TURKEYSMS,
                    'api_link'        => 'https://turkeysms.com.tr/api/v3/get/get.php',
                    'otp_api_link'    => 'https://turkeysms.com.tr/api/v3/otp/otp_get.php',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ESENDEX => [
                    'name'            => SendingServer::TYPE_ESENDEX,
                    'settings'        => SendingServer::TYPE_ESENDEX,
                    'api_link'        => 'https://www.esendex.com/secure/messenger/formpost/SendSMS.aspx',
                    'username'        => 'Username/Email',
                    'password'        => 'API Password',
                    'account_sid'     => 'Account ID',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_RUBSMSRU => [
                    'name'            => SendingServer::TYPE_RUBSMSRU,
                    'settings'        => SendingServer::TYPE_RUBSMSRU,
                    'auth_link'       => 'https://lk-1rub-sms.ru/api/auth/token/',
                    'api_link'        => 'https://lk-1rub-sms.ru/api/distribution/start_distribution_v2/',
                    'username'        => 'Username/Email',
                    'password'        => 'Password',
                    'c1'              => 'Company',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSHUBAO => [
                    'name'            => SendingServer::TYPE_SMSHUBAO,
                    'settings'        => SendingServer::TYPE_SMSHUBAO,
                    'api_link'        => 'https://app.smshub.ao/api/sendsms',
                    'auth_id'         => 'Auth ID',
                    'api_secret'      => 'secret_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXTBACK => [
                    'name'            => SendingServer::TYPE_TEXTBACK,
                    'settings'        => SendingServer::TYPE_TEXTBACK,
                    'api_link'        => 'https://api.textback.ai/api/v2/send/sendText',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_OMBALA => [
                    'name'            => SendingServer::TYPE_OMBALA,
                    'settings'        => SendingServer::TYPE_OMBALA,
                    'api_link'        => 'https://api.useombala.ao/v1/messages',
                    'api_token'       => 'API Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TEXTHUB => [
                    'name'            => SendingServer::TYPE_TEXTHUB,
                    'settings'        => SendingServer::TYPE_TEXTHUB,
                    'api_link'        => 'https://api.texthub.com/v2/',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_VERIFYSMSURWAY => [
                    'name'            => SendingServer::TYPE_VERIFYSMSURWAY,
                    'settings'        => SendingServer::TYPE_VERIFYSMSURWAY,
                    'auth_link'       => 'https://verify.smsurway.com.ng/api/v1/login',
                    'otp_api_link'    => 'https://verify.smsurway.com.ng/api/v1/send_otp',
                    'c1'              => 'Email',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => true,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_FASTSMS101 => [
                    'name'            => SendingServer::TYPE_FASTSMS101,
                    'settings'        => SendingServer::TYPE_FASTSMS101,
                    'api_link'        => 'https://panel2.fastsms101.com/smsapi',
                    'username'        => 'User',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_WHATAPICLOUD => [
                    'name'            => SendingServer::TYPE_WHATAPICLOUD,
                    'settings'        => SendingServer::TYPE_WHATAPICLOUD,
                    'api_link'        => 'https://gate.whapi.cloud/messages/text',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_BIRD => [
                    'name'            => SendingServer::TYPE_BIRD,
                    'settings'        => SendingServer::TYPE_BIRD,
                    'api_link'        => 'https://api.bird.com/workspaces/{workspace_id}/channels/channel_id/calls',
                    'access_key'      => 'access_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => true,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MOZESMS => [
                    'name'            => SendingServer::TYPE_MOZESMS,
                    'settings'        => SendingServer::TYPE_MOZESMS,
                    'api_link'        => 'https://api.mozesms.com/message/v2',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MESSAGECENTRAL => [
                    'name'            => SendingServer::TYPE_MESSAGECENTRAL,
                    'settings'        => SendingServer::TYPE_MESSAGECENTRAL,
                    'auth_link'       => 'https://cpaas.messagecentral.com/auth/v1/authentication/token',
                    'api_link'        => 'https://cpaas.messagecentral.com/verification/v3/send',
                    'auth_token'      => 'auth_token',
                    'c1'              => 'Customer ID',
                    'c2'              => 'Email',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_PHDSMS => [
                    'name'            => SendingServer::TYPE_PHDSMS,
                    'settings'        => SendingServer::TYPE_PHDSMS,
                    'api_link'        => 'http://54.169.201.159:9511/api/send-sms-single',
                    'c1'              => 'SP ID',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MDSMS => [
                    'name'            => SendingServer::TYPE_MDSMS,
                    'settings'        => SendingServer::TYPE_MDSMS,
                    'api_link'        => 'https://openapi.mdtechcorp.com:20000/openapi',
                    'username'        => 'username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DECISIONTELECOM => [
                    'name'            => SendingServer::TYPE_DECISIONTELECOM,
                    'settings'        => SendingServer::TYPE_DECISIONTELECOM,
                    'api_link'        => 'https://web.it-decision.com/v1/api/send-sms',
                    'api_key'         => 'api_key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_MESSAGINGSERVICECOTZ => [
                    'name'            => SendingServer::TYPE_MESSAGINGSERVICECOTZ,
                    'settings'        => SendingServer::TYPE_MESSAGINGSERVICECOTZ,
                    'api_link'        => 'https://messaging-service.co.tz/api/sms/v1/text/single',
                    'username'        => 'username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_CLOSUM => [
                    'name'            => SendingServer::TYPE_CLOSUM,
                    'settings'        => SendingServer::TYPE_CLOSUM,
                    'api_link'        => 'https://api.closum.com/v2/sms/send/',
                    'api_key'         => 'Api Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_EGOSMS => [
                    'name'            => SendingServer::TYPE_EGOSMS,
                    'settings'        => SendingServer::TYPE_EGOSMS,
                    'api_link'        => 'https://www.egosms.co/api/v1/json/',
                    'username'        => 'username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_ANKAREXIO => [
                    'name'            => SendingServer::TYPE_ANKAREXIO,
                    'settings'        => SendingServer::TYPE_ANKAREXIO,
                    'api_link'        => 'https://app.ankarex.io/api/json',
                    'api_token'       => 'Api Token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

//                SendingServer::TYPE_HOLLYRENEW => [
//                    'name'            => SendingServer::TYPE_HOLLYRENEW,
//                    'settings'        => SendingServer::TYPE_HOLLYRENEW,
//                    'api_link'        => 'https://sms2.hollyrenew.website/apiV2',
//                    'api_token'       => 'Api Token',
//                    'schedule'        => true,
//                    'type'            => 'http',
//                    'two_way'         => false,
//                    'plain'           => true,
//                    'mms'             => false,
//                    'voice'           => false,
//                    'whatsapp'        => false,
//                    'viber'           => false,
//                    'otp'             => false,
//                    'sms_per_request' => 1,
//                    'quota_value'     => 500,
//                    'quota_base'      => 1,
//                    'quota_unit'      => 'minute',
//                ],
//
//                SendingServer::TYPE_FORTYTWO => [
//                    'name'            => SendingServer::TYPE_FORTYTWO,
//                    'settings'        => SendingServer::TYPE_FORTYTWO,
//                    'api_link'        => 'https://rest.fortytwo.com/1/im',
//                    'api_key'         => 'API Key',
//                    'sms_type'        => 'Promotional',
//                    'schedule'        => true,
//                    'type'            => 'http',
//                    'two_way'         => true,
//                    'plain'           => true,
//                    'mms'             => true,
//                    'voice'           => true,
//                    'whatsapp'        => false,
//                    'viber'           => true,
//                    'otp'             => true,
//                    'sms_per_request' => 1,
//                    'quota_value'     => 500,
//                    'quota_base'      => 1,
//                    'quota_unit'      => 'minute',
//                ],


                SendingServer::TYPE_NEXAH => [
                    'name'            => SendingServer::TYPE_NEXAH,
                    'settings'        => SendingServer::TYPE_NEXAH,
                    'api_link'        => 'https://smsvas.com/bulk/public/index.php/api/v1/sendsms',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_GENNET => [
                    'name'            => SendingServer::TYPE_GENNET,
                    'settings'        => SendingServer::TYPE_GENNET,
                    'api_link'        => 'https://gbarta.gennet.com.bd/api/v1/smsapi',
                    'api_key'         => 'API Key',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_DURBAR => [
                    'name'            => SendingServer::TYPE_DURBAR,
                    'settings'        => SendingServer::TYPE_DURBAR,
                    'api_link'        => 'https://sms.durbar71.com/httpapi/sendsms',
                    'c1'              => 'User ID',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_SMSCOUNTRY_STANDARD => [
                    'name'            => SendingServer::TYPE_SMSCOUNTRY_STANDARD,
                    'settings'        => SendingServer::TYPE_SMSCOUNTRY_STANDARD,
                    'api_link'        => 'https://api.smscountry.com/SMSCwebservice_bulk.aspx',
                    'username'        => 'Username',
                    'password'        => 'Password',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],

                SendingServer::TYPE_TWILIO_WHATSAPP_BY_TEMPLATE => [
                    'name'            => SendingServer::TYPE_TWILIO_WHATSAPP_BY_TEMPLATE,
                    'settings'        => SendingServer::TYPE_TWILIO_WHATSAPP_BY_TEMPLATE,
                    'account_sid'     => 'account_sid',
                    'auth_token'      => 'auth_token',
                    'c1'              => 'messagingServiceSid',
                    'schedule'        => true,
                    'type'            => 'whatsapp',
                    'two_way'         => false,
                    'plain'           => false,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => true,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],


                SendingServer::TYPE_EASIFY => [
                    'name'            => SendingServer::TYPE_EASIFY,
                    'settings'        => SendingServer::TYPE_EASIFY,
                    'api_link'        => 'https://backend.easify.app/api/v2/sms/send',
                    'api_token'       => 'api_token',
                    'schedule'        => true,
                    'type'            => 'http',
                    'two_way'         => false,
                    'plain'           => true,
                    'mms'             => false,
                    'voice'           => false,
                    'whatsapp'        => false,
                    'viber'           => false,
                    'otp'             => false,
                    'sms_per_request' => 1,
                    'quota_value'     => 500,
                    'quota_base'      => 1,
                    'quota_unit'      => 'minute',
                ],


            ];

        }

        /**
         * @throws GeneralException
         */
        private function save(SendingServer $sendingServer): bool
        {
            if ( ! $sendingServer->save()) {
                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            }

            return true;
        }

        /**
         * @throws GeneralException
         */
        public function destroy(SendingServer $sendingServer, int $user_id = null): bool
        {

            User::where('api_sending_server', $sendingServer->id)->update(['api_sending_server' => null]);

            if ( ! $sendingServer->delete()) {
                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            }

            return true;
        }

        /**
         * @return mixed
         *
         * @throws Throwable
         */
        public function batchDestroy(array $ids, int $user_id = null): bool
        {

            DB::transaction(function () use ($ids) {
                // This won't call eloquent events, change to destroy if needed
                if ($this->query()->whereIn('uid', $ids)->delete()) {
                    return true;
                }

                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            });

            return true;
        }

        /**
         * @return mixed
         *
         * @throws Exception|Throwable
         */
        public function batchActive(array $ids): bool
        {
            DB::transaction(function () use ($ids) {
                if ($this->query()->whereIn('uid', $ids)
                    ->update(['status' => true])
                ) {
                    return true;
                }

                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            });

            return true;
        }

        /**
         * @throws Exception|Throwable
         * @throws Exception
         */
        public function update(SendingServer $sendingServer, array $input): SendingServer
        {
            if ( ! $sendingServer->update($input)) {
                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            }

            return $sendingServer;
        }

        /**
         * @return mixed
         *
         * @throws Exception|Throwable
         */
        public function batchDisable(array $ids): bool
        {
            DB::transaction(function () use ($ids) {
                if ($this->query()->whereIn('uid', $ids)
                    ->update(['status' => false])
                ) {
                    return true;
                }

                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            });

            return true;
        }

        public function sendTestSMS(SendingServer $sendingServer, array $input)
        {
            // TODO: Implement sendTestSMS() method.
        }

        /**
         * Store custom sending server
         *
         *
         * @throws GeneralException
         */
        public function storeCustom(array $input): SendingServer
        {
            $sendingServerInput = ['name',
                'api_link',
                'success_keyword',
                'plain',
                'schedule',
                'quota_value',
                'quota_base',
                'quota_unit',
                'sms_per_request',
                'cutting_value',
                'cutting_unit',
                'cutting_logic'];

            /** @var SendingServer $sendingServer */
            $sendingServer = $this->make(Arr::only($input, $sendingServerInput));

            $settings = ucfirst(preg_replace('/\s+/', '', $input['name']));

            $sendingServer->settings = $settings;
            $sendingServer->status   = true;
            $sendingServer->custom   = true;
            $sendingServer->type     = 'http';
            $sendingServer->user_id  = auth()->user()->id;

            if ($this->save($sendingServer)) {

                $customServer        = new CustomSendingServer();
                $customSendingServer = $customServer->make(Arr::only($input, ['http_request_method',
                    'json_encoded_post',
                    'content_type',
                    'content_type_accept',
                    'character_encoding',
                    'ssl_certificate_verification',
                    'authorization',
                    'multi_sms_delimiter',
                    'username_param',
                    'username_value',
                    'password_param',
                    'password_value',
                    'password_status',
                    'action_param',
                    'action_value',
                    'action_status',
                    'source_param',
                    'source_value',
                    'source_status',
                    'destination_param',
                    'message_param',
                    'unicode_param',
                    'unicode_value',
                    'unicode_status',
                    'route_param',
                    'route_value',
                    'route_status',
                    'language_param',
                    'language_value',
                    'language_status',
                    'custom_one_param',
                    'custom_one_value',
                    'custom_one_status',
                    'custom_two_param',
                    'custom_two_value',
                    'custom_two_status',
                    'custom_three_param',
                    'custom_three_value',
                    'custom_three_status']));

                $customSendingServer->server_id = $sendingServer->id;

                if ( ! $this->saveCustom($customSendingServer)) {
                    $sendingServer->delete();
                    throw new GeneralException(__('locale.exceptions.something_went_wrong'));
                }

                return $sendingServer;
            }
            throw new GeneralException(__('locale.exceptions.something_went_wrong'));
        }

        /**
         * @throws GeneralException
         */
        private function saveCustom(CustomSendingServer $customSendingServer): bool
        {
            if ( ! $customSendingServer->save()) {
                throw new GeneralException(__('locale.exceptions.something_went_wrong'));
            }

            return true;
        }

        /**
         * @throws Exception|Throwable
         * @throws Exception
         */
        public function updateCustom(SendingServer $sendingServer, array $input): SendingServer
        {

            if ($sendingServer->update($input)) {
                $customServer = CustomSendingServer::where('server_id', $sendingServer->id)->first();

                if ( ! $customServer->update($input)) {
                    throw new GeneralException(__('locale.exceptions.something_went_wrong'));
                }

                return $sendingServer;
            }

            throw new GeneralException(__('locale.exceptions.something_went_wrong'));
        }


        /**
         * @return true
         * @throws Throwable
         */
        public function batchCoverageEnable(SendingServer $sendingServer, array $ids)
        {
            $batchEnable = SendingServerBasedPricingPlans::where('sending_server', $sendingServer->id)->whereIn('uid', $ids)->update([
                'status' => true,
            ]);

            if ($batchEnable) {
                return true;
            }

            throw new GeneralException(__('locale.exceptions.something_went_wrong'));
        }

        /**
         * @return true
         * @throws GeneralException
         */
        public function batchCoverageDisable(SendingServer $sendingServer, array $ids)
        {

            $batchDisable = SendingServerBasedPricingPlans::where('sending_server', $sendingServer->id)->whereIn('uid', $ids)->update([
                'status' => false,
            ]);

            if ($batchDisable) {
                return true;
            }

            throw new GeneralException(__('locale.exceptions.something_went_wrong'));
        }

        /**
         * @return true
         * @throws GeneralException
         */
        public function batchCoverageDelete(SendingServer $sendingServer, array $ids)
        {
            $batchDelete = SendingServerBasedPricingPlans::where('sending_server', $sendingServer->id)->whereIn('uid', $ids)->delete();

            if ($batchDelete) {
                return true;
            }

            throw new GeneralException(__('locale.exceptions.something_went_wrong'));
        }

    }
